<?php

if ( !defined( 'ABSPATH' ) ) {
	exit;
}

if ( !class_exists( 'WCRP_Rental_Products_Order_Line_Items' ) ) {

	class WCRP_Rental_Products_Order_Line_Items {

		public function __construct() {

			add_action( 'admin_footer', array( $this, 'order_line_item_scripts' ) );
			add_filter( 'woocommerce_admin_html_order_item_class', array( $this, 'order_line_item_row_class' ), 10, 3 );
			add_action( 'woocommerce_checkout_create_order_line_item', array( $this, 'order_line_item_data_create' ), 10, 4 );
			add_filter( 'woocommerce_order_item_display_meta_key', array( $this, 'order_line_item_display_meta_key' ) );
			add_filter( 'woocommerce_order_item_display_meta_value', array( $this, 'order_line_item_display_meta_value' ), 10, 3 );
			add_filter( 'woocommerce_order_item_get_formatted_meta_data', array( $this, 'order_line_item_formatted_meta_data' ), 10, 2 );
			add_action( 'woocommerce_after_order_itemmeta', array( $this, 'order_line_item_actions' ), 10, 3 );
			add_action( 'woocommerce_before_delete_order_item', array( $this, 'order_line_item_delete' ) );
			add_action( 'woocommerce_order_item_add_line_buttons', array( $this, 'order_line_item_add_line_buttons' ) );
			add_filter( 'woocommerce_prevent_adjust_line_item_product_stock', array( $this, 'order_line_item_restrict_purchasable_stock_adjustment_if_rental' ), 10, 3 );

		}

		public function order_line_item_scripts() {

			global $post;

			if ( !empty( $post ) ) { // Global post may not exist in some areas of the admin such as custom menu pages and would trigger a PHP warning

				if ( 'shop_order' == get_post_type( $post->ID ) ) {

					// Note that there isn't a check for if the order contains rentals here using wcrp_rental_products_order_has_rentals() or similar, there used to be one however this stops the script below running if no rentals, and now rental products can be added to orders through the dashboard a new order doesn't initially have any rentals, so the scripts below wouldn't run - but they are needed if a rental product is added to the order, if the old condition was still here it would mean the order line item buttons wouldn't function correctly until the page is refreshed after adding a rental product (and the other things in the script below)

					?>

					<script>
						jQuery( document ).ready( function( $ ) {

							// Disable rental order item meta fields

							function disableRentalOrderItemMetaFields() {

								$( '.wcrp-rental-products-order-item-is-rental .name .edit input[type="text"]' ).each( function( index ) {

									if ( $( this ).val().startsWith( 'wcrp_rental_products_' ) ) {

										$( this ).attr( 'readonly', true );
										$( this ).parent().find( 'textarea' ).attr( 'readonly', true );  // Disable order item meta fields, readonly instead of disabled to ensures still submitted
										$( this ).closest( 'tr' ).find( '.remove_order_item_meta' ).attr( 'disabled', true ); // Disable deleting order item meta
										$( this ).closest( '.wcrp-rental-products-order-item-is-rental' ).find( '.quantity .edit .quantity' ).attr( 'readonly', true ); // Quantity fields on edit/refund, readonly instead of disabled to ensures still submitted

									}

								});

							}

							// Mark as returned

							$( document ).on( 'click', '.wcrp-rental-products-line-item-returned', function( e ) {

								if ( !confirm( "<?php esc_html_e( 'Are you sure you want to mark this rental as returned?', 'wcrp-rental-products' ); ?>" ) ) {

									e.preventDefault();

								}

							});

							// Change rental

							$( document ).on( 'click', '.wcrp-rental-products-line-item-change-rental', function( e ) {

								e.preventDefault();

								alert( "<?php esc_html_e( 'The dates and quantities for this rental are reserved, if you wish to change this rental it will need to be cancelled using the cancel rental button to make the reserved dates and quantities available again.', 'wcrp-rental-products' ); ?>" + "\n\n" + "<?php esc_html_e( 'Once cancelled use the add item(s) button to add the product again - note that the rental dates and quantities available for selection are subject to the same availability shown on the product page. Depending on the changes made you may need to take an additional payment or add a refund.', 'wcrp-rental-products' ); ?>" );

							});

							// Cancel rental

							$( document ).on( 'click', '.wcrp-rental-products-line-item-cancel-rental', function( e ) {

								if ( !confirm( "<?php esc_html_e( 'Are you sure you want to cancel this rental?', 'wcrp-rental-products' ); ?>" ) ) {

									e.preventDefault();

								}

							});

							// Restrict restock refunded items

							function restrictRestockRefundedItems() {

								var restrictRestockRefundedItems = false;

								$( '#wcrp-rental-products-restock-refunded-items-notice' ).remove();

								$( '#order_line_items .item' ).each( function( index ) {

									if ( $( this ).hasClass( 'wcrp-rental-products-order-item-is-rental' ) ) {

										if ( $( this ).find( '.refund_order_item_qty' ).val() > 0 ) {

											restrictRestockRefundedItems = true;

										}

									}

								});

								if ( restrictRestockRefundedItems == true ) {

									// If rental products are included in the order then the restocking refunded items option is disabled, this is because an order could contain a rental or purchase product (we don't have a means of checking if a rental based order line item is a rental or purchase to only disable for that scenario, getting the post meta of that product wouldn't be reliable as it could have changed recently), if it's the rental of a rental or purchase product if this option was enabled it would increase the purchase stock level based off _reduced_stock order item meta, there isn't a way to stop this occuring due to wc_restock_refunded_items() works with a limitation of not being able to stop it occuring due to how we need _reduced_stock to be there to ensure purchase stock doesn't get reduced initially, this being restricted is related to reduced_stock_meta_manipulation() and order_line_item_restrict_purchasable_stock_adjustment_if_rental() functions, review these for further clarity

									$( '#restock_refunded_items' ).prop( 'checked', false ).attr( 'disabled', true );

									$( '<div id="wcrp-rental-products-restock-refunded-items-notice"><?php esc_html_e( 'Restock refunded items disabled as refund includes rental items', 'wcrp-rental-products' ); ?><br><?php esc_html_e( 'Rental stock reserved is made available upon marking as returned or cancelled after refund', 'wcrp-rental-products' ); ?><br><?php esc_html_e( 'To refund and restock non-rental items refund these seperately to rental items', 'wcrp-rental-products' ); ?></div>' ).appendTo( 'label[for="restock_refunded_items"]' );

								} else {

									$( '#restock_refunded_items' ).attr( 'disabled', false );

								}

							}

							// Initial

							disableRentalOrderItemMetaFields();
							restrictRestockRefundedItems();

							// On AJAX complete (ensures if any changes made via AJAX request the fields get set to what they should such as disabling fields form edit, e.g. editing an order line item and clicking the cancel button would send an AJAX request and replace the order line items markup, after that happens this re-runs the functions ensuring the fields are as required)

							$( document ).ajaxComplete( function() {

								disableRentalOrderItemMetaFields();
								restrictRestockRefundedItems();

							});

							// On change of order item refund quantity

							$( document ).on( 'change', '.refund_order_item_qty', function( e ) {

								restrictRestockRefundedItems();

							});

						});

					</script>

					<?php

				}

			}

		}

		public function order_line_item_row_class( $class, $item, $order ) {

			$item_id = $item->get_id();

			if ( !empty( $item_id ) ) {

				if ( !empty( wc_get_order_item_meta( $item->get_id(), 'wcrp_rental_products_rent_from', true ) ) || !empty( wc_get_order_item_meta( $item->get_id(), 'wcrp_rental_products_cancelled', true ) ) ) { // Cancelled condition means it still has a rental class even if cancelled

					$class .= ' wcrp-rental-products-order-item-is-rental '; // Spaces added around incase other extensions use this and lead on without spaces, would give it an incorrect class

				}

			}

			return $class;

		}

		public function order_line_item_data_create( $item, $cart_item_key, $values, $order ) {

			if ( isset( $values['wcrp_rental_products_rent_from'] ) && isset( $values['wcrp_rental_products_rent_to'] ) && isset( $values['wcrp_rental_products_return_days_threshold'] ) ) {

				// '' !== checks as could be 0 which would be conditioned as empty if using !empty

				if ( '' !== $values['wcrp_rental_products_rent_from'] && '' !== $values['wcrp_rental_products_rent_to'] && '' !== $values['wcrp_rental_products_return_days_threshold'] ) {

					$item->add_meta_data( 'wcrp_rental_products_rent_from', $values['wcrp_rental_products_rent_from'] );
					$item->add_meta_data( 'wcrp_rental_products_rent_to', $values['wcrp_rental_products_rent_to'] );
					$item->add_meta_data( 'wcrp_rental_products_return_days_threshold', $values['wcrp_rental_products_return_days_threshold'] ); // On the order screen the return days threshold is conditionally displayed/hidden via order_line_item_formatted_meta_data()

					// In person pick up/return meta data, these are all added so exist in the order line item data, but only some are displayed

					if ( isset( $values['wcrp_rental_products_in_person_pick_up_return'] ) ) {

						if ( 'yes' == $values['wcrp_rental_products_in_person_pick_up_return'] ) {

							$item->add_meta_data( 'wcrp_rental_products_in_person_pick_up_return', sanitize_text_field( $values['wcrp_rental_products_in_person_pick_up_return'] ) );
							$item->add_meta_data( 'wcrp_rental_products_in_person_pick_up_date', sanitize_text_field( $values['wcrp_rental_products_in_person_pick_up_date'] ) );
							$item->add_meta_data( 'wcrp_rental_products_in_person_pick_up_time', sanitize_text_field( $values['wcrp_rental_products_in_person_pick_up_time'] ) );
							$item->add_meta_data( 'wcrp_rental_products_in_person_pick_up_fee', sanitize_text_field( $values['wcrp_rental_products_in_person_pick_up_fee'] ) );
							$item->add_meta_data( 'wcrp_rental_products_in_person_return_date', sanitize_text_field( $values['wcrp_rental_products_in_person_return_date'] ) );
							$item->add_meta_data( 'wcrp_rental_products_in_person_return_date_type', sanitize_text_field( $values['wcrp_rental_products_in_person_return_date_type'] ) );
							$item->add_meta_data( 'wcrp_rental_products_in_person_return_time', sanitize_text_field( $values['wcrp_rental_products_in_person_return_time'] ) );
							$item->add_meta_data( 'wcrp_rental_products_in_person_return_fee', sanitize_text_field( $values['wcrp_rental_products_in_person_return_fee'] ) );

						}

					}

				}

			}

		}

		public function order_line_item_display_meta_key( $display_key ) {

			if ( 'wcrp_rental_products_rent_from' == $display_key  ) {

				$display_key = apply_filters( 'wcrp_rental_products_text_rent_from', get_option( 'wcrp_rental_products_text_rent_from' ) );

			} elseif ( 'wcrp_rental_products_rent_to' == $display_key ) {

				$display_key = apply_filters( 'wcrp_rental_products_text_rent_to', get_option( 'wcrp_rental_products_text_rent_to' ) );

			} elseif ( 'wcrp_rental_products_return_days_threshold' == $display_key ) {

				$display_key = apply_filters( 'wcrp_rental_products_text_rental_return_within', get_option( 'wcrp_rental_products_text_rental_return_within' ) ); // Conditionally displayed/hidden via order_line_item_formatted_meta_data()

			} elseif ( 'wcrp_rental_products_returned' == $display_key ) {

				$display_key = apply_filters( 'wcrp_rental_products_text_rental_returned', get_option( 'wcrp_rental_products_text_rental_returned' ) );

			} elseif ( 'wcrp_rental_products_cancelled' == $display_key ) {

				$display_key = apply_filters( 'wcrp_rental_products_text_rental_cancelled', get_option( 'wcrp_rental_products_text_rental_cancelled' ) );

			} elseif ( 'wcrp_rental_products_in_person_pick_up_return' == $display_key ) { // In person pick up/return display keys start here, only the keys which are displayed are included, the ones not omitted via order_line_item_formatted_meta_data()

				$display_key = apply_filters( 'wcrp_rental_products_text_in_person_pick_up_return', get_option( 'wcrp_rental_products_text_in_person_pick_up_return' ) );

			} elseif ( 'wcrp_rental_products_in_person_pick_up_time' == $display_key ) {

				$display_key = apply_filters( 'wcrp_rental_products_text_pick_up_time', get_option( 'wcrp_rental_products_text_pick_up_time' ) );

			} elseif ( 'wcrp_rental_products_in_person_return_date' == $display_key ) {

				$display_key = apply_filters( 'wcrp_rental_products_text_return_date', get_option( 'wcrp_rental_products_text_return_date' ) ); // This is only shown if return date !== rent to date via order_line_item_formatted_meta_data()

			} elseif ( 'wcrp_rental_products_in_person_return_time' == $display_key ) {

				$display_key = apply_filters( 'wcrp_rental_products_text_return_time', get_option( 'wcrp_rental_products_text_return_time' ) );

			}

			return $display_key;

		}

		public function order_line_item_display_meta_value( $display_meta_value, $meta, $order_item ) {

			if ( !empty( $meta ) ) {

				// Format meta values if needed to be displayed differently from how stored

				if ( 'wcrp_rental_products_rent_from' == $meta->key || 'wcrp_rental_products_rent_to' == $meta->key ) {

					$display_meta_value = date_i18n( wcrp_rental_products_rental_date_format(), strtotime( $display_meta_value ) ); // Displays date in format required

				} elseif ( 'wcrp_rental_products_returned' == $meta->key ) {

					$display_meta_value = ucfirst( $display_meta_value ); // Changes yes to Yes

				} elseif ( 'wcrp_rental_products_return_days_threshold' == $meta->key ) {

					$display_meta_value = $display_meta_value . ' ' . esc_html__( 'days', 'wcrp-rental-products' ) . ' ' . esc_html__( '(', 'wcrp-rental-products' ) . date_i18n( wcrp_rental_products_rental_date_format(), strtotime( gmdate( 'Y-m-d', strtotime( wc_get_order_item_meta( $order_item->get_id(), 'wcrp_rental_products_rent_to', true ) . ' + ' . wc_get_order_item_meta( $order_item->get_id(), 'wcrp_rental_products_return_days_threshold', true ) . 'days' ) ) ) ) . esc_html__( ')', 'wcrp-rental-products' ); // Adds days and date to end of day number, same as cart items

				} elseif ( 'wcrp_rental_products_cancelled' == $meta->key ) {

					$display_meta_value = ucfirst( $display_meta_value ); // Changes yes to Yes

				} elseif ( 'wcrp_rental_products_in_person_pick_up_return' == $meta->key ) { // In person pick up/return meta values start here, only the values which are displayed are included, the ones not omitted via order_line_item_formatted_meta_data()

					$display_meta_value = ucfirst( $display_meta_value ); // Changes yes to Yes

				} elseif ( 'wcrp_rental_products_in_person_pick_up_time' == $meta->key ) {

					$display_meta_value = WCRP_Rental_Products_Misc::four_digit_time_formatted( $display_meta_value ); // Displays time in format required

				} elseif ( 'wcrp_rental_products_in_person_return_date' == $meta->key ) {

					$display_meta_value = date_i18n( wcrp_rental_products_rental_date_format(), strtotime( $display_meta_value ) ); // Displays date in format required, this is only shown if return date !== rent to date via order_line_item_formatted_meta_data()

				} elseif ( 'wcrp_rental_products_in_person_return_time' == $meta->key ) {

					$display_meta_value = WCRP_Rental_Products_Misc::four_digit_time_formatted( $display_meta_value ); // Displays time in format required

				}

			}

			return $display_meta_value;

		}

		public function order_line_item_formatted_meta_data( $formatted_meta_data, $item ) {

			if ( !empty( $formatted_meta_data ) ) {

				// Conditionally hide meta where we do not want it to display

				foreach ( $formatted_meta_data as $key => $meta ) {

					if ( 'wcrp_rental_products_return_days_threshold' == $meta->key ) {

						if ( '0' == $meta->value ) { // Return days threshold only ever shown if > 0

							unset( $formatted_meta_data[$key] );

						} else {

							if ( 'no' == get_option( 'wcrp_rental_products_return_days_display' ) ) { // If return days display disabled then return days should not display to the customer

								// The anonymous function below fakes the order item meta display for return days threshold so it is still shown in the dashboard, important: we used to simply use a !is_admin() conditional for only disabling return days display on the frontend, however we found that if a store admin is editing orders and changing order statuses because this is an admin dashboard page it meant that the return days threshold was being included on customer facing emails, hence why we fake the order item meta display below, then unset the order item meta afterwards, the faked order item meta is not included in emails as based off the woocommerce_after_order_itemmeta action hook not used in emails

								add_action( 'woocommerce_after_order_itemmeta', function( $passed_item_id, $passed_item, $passed_product ) use ( $formatted_meta_data, $key, $item ) {

									// The passed_ prefixed variables are the ones available from woocommerce_after_order_itemmeta, the use based variables are passed from this function (order_line_item_formatted_meta_data) itself, we then can compare the 2 to ensure it only gets displayed once (see condition and comment below) and then use the existing $formatted_meta_data instead of recreating it

									if ( $passed_item == $item ) { // Ensures that it only gets added to the line item it should be on, without this if there are 2+ order line items it will output the first and second order line item return days threshold on the second order line item and so on

										echo '<table cellspacing="0" class="display_meta" style="margin-top: 0;"><tbody><tr><th>' . wp_kses_post( $formatted_meta_data[$key]->display_key ) . esc_html__( ':', 'wcrp-rental-products' ) . '</th><td>' . wp_kses_post( $formatted_meta_data[$key]->display_value ) . '</td></tr></tbody></table>';

									}

								}, 0, 3 ); // 0 priority so before order_line_item_actions() and any other custom stuff

								unset( $formatted_meta_data[$key] ); // Unset occurs after the anonymous function above as that requires $formatted_meta_data[$key] to still be available

							}

						}

					} elseif ( 'wcrp_rental_products_in_person_pick_up_date' == $meta->key ) {

						unset( $formatted_meta_data[$key] ); // Not shown as always same as rent from date

					} elseif ( 'wcrp_rental_products_in_person_pick_up_fee' == $meta->key ) {

						unset( $formatted_meta_data[$key] ); // Not shown as added to item total

					} elseif ( 'wcrp_rental_products_in_person_return_date' == $meta->key ) {

						if ( $meta->value == $item->get_meta( 'wcrp_rental_products_rent_to' ) ) {

							unset( $formatted_meta_data[$key] ); // Not shown if return date is same as rent to date

						}

					} elseif ( 'wcrp_rental_products_in_person_return_date_type' == $meta->key ) {

						unset( $formatted_meta_data[$key] ); // Not shown as just used for availability checks

					} elseif ( 'wcrp_rental_products_in_person_return_fee' == $meta->key ) {

						unset( $formatted_meta_data[$key] ); // Not shown as added to item total

					}

				}

			}

			return $formatted_meta_data;

		}

        public function order_line_item_actions($item_id, $item, $product)
        {

            $order_id = $item->get_order_id();
            $user_id = wc_get_order($order_id)->get_user_id();
            $boxes = get_user_meta($user_id, 'boxes', true);
            $is_box_line = false;
            foreach ($boxes as $box) {
                if (isset($box->items[$item_id])) {
                    $is_box_line = true;
                }
            }

            $rent_from = wc_get_order_item_meta($item_id, 'wcrp_rental_products_rent_from', true);
            $rent_to = wc_get_order_item_meta($item_id, 'wcrp_rental_products_rent_to', true);

            // No check for if product is a rental because the product might have been a rental but no longer is
            // No check on order status completed here because if it's completed the order items are marked as returned, so the returned condition below is in effect a completed order status condition too
            if (!empty($rent_from) && !empty($rent_to)) { // This also ensures if cancelled the line item actions do not show (as from/to dates meta they will have been removed)
                $returned = wc_get_order_item_meta($item_id, 'wcrp_rental_products_returned', true);
                ?>
                <div class="wcrp-rental-products-line-item-actions">
                    <?php if ('yes' !== $returned) { ?>
                        <?php if (!$is_box_line) { ?>
                            <button type="submit" name="wcrp_rental_products_line_item_returned"
                                    class="wcrp-rental-products-line-item-returned button button-small button-primary"
                                    value="<?php echo esc_html($item_id); ?>"><?php esc_html_e('Mark as returned', 'wcrp-rental-products'); ?></button>
                        <?php } ?>
                        <button type="submit" name="wcrp_rental_products_line_item_edit_rental"
                                class="wcrp-rental-products-line-item-change-rental button button-small"
                                value="<?php echo esc_html($item_id); ?>"><?php esc_html_e('Change rental', 'wcrp-rental-products'); ?></button>
                        <button type="submit" name="wcrp_rental_products_line_item_cancel_rental"
                                class="wcrp-rental-products-line-item-cancel-rental button button-small"
                                value="<?php echo esc_html($item_id); ?>"><?php esc_html_e('Cancel rental', 'wcrp-rental-products'); ?></button>
                    <?php } ?>
                </div>
                <?php
            }
        }

		public function order_line_item_delete( $order_item_id ) {

			global $wpdb;

			$wpdb->query(
				$wpdb->prepare(
					"DELETE FROM `{$wpdb->prefix}wcrp_rental_products_rentals` WHERE `order_item_id` = %d;",
					$order_item_id
				)
			);

			$wpdb->query(
				$wpdb->prepare(
					"DELETE FROM `{$wpdb->prefix}wcrp_rental_products_rentals_archive` WHERE `order_item_id` = %d;",
					$order_item_id
				)
			);

		}

		public function order_line_item_add_line_buttons( $order ) {

			global $pagenow;
			global $post;

			require_once ABSPATH . 'wp-admin/includes/plugin.php';

			if ( is_plugin_active( 'woocommerce-product-addons/woocommerce-product-addons.php' ) ) {

				$addons = 'yes';

			} else {

				$addons = 'no';

			}

			if ( is_plugin_active( 'deposits-partial-payments-for-woocommerce/start.php' ) || is_plugin_active( 'deposits-partial-payments-for-woocommerce-pro/start.php' ) ) {

				$deposits_partial_payments = 'yes';

			} else {

				$deposits_partial_payments = 'no';
				
			}

			$new_order = 'no';

			if ( isset( $post ) ) {

				if ( 'post-new.php' == $pagenow ) {

					if ( 'shop_order' == get_post_type( $post->ID ) ) {

						$new_order = 'yes';

					}

				}

			} elseif ( isset( $_POST['security'] ) && wp_verify_nonce( sanitize_key( $_POST['security'] ), 'order-item' ) ) {

				// These conditions are here because an order can be saved via order line item actions save button (however this does not create the order), in this scenario $pagenow and $post is not populated as it's an AJAX request, so we get the order id from $_POST of the AJAX request not $post->ID, we don't need 

				if ( isset( $_POST['action'] ) && isset( $_POST['order_id'] ) ) {

					if ( 'woocommerce_save_order_items' == sanitize_text_field( $_POST['action'] ) ) {

						if ( 'auto-draft' == get_post_status( sanitize_text_field( $_POST['order_id'] ) ) ) { // Orders not yet created have the auto-draft post status, we use this as cannot check if $pagenow is post-new.php as an AJAX request

							$new_order = 'yes';

						}

					}

				}				

			}

			echo '<button type="button" class="button" id="wcrp-rental-products-add-rental-products" data-addons="' . esc_attr( $addons ) . '" data-deposits-partial-payments="' . esc_attr( $deposits_partial_payments ) . '" data-new-order="' . esc_attr( $new_order ) . '" data-url="' . esc_url( get_site_url() ) . '">' . esc_html__( 'Add rental product(s)', 'wcrp-rental-products' ) . '</button>';

		}

		public function order_line_item_restrict_purchasable_stock_adjustment_if_rental( $restrict, $order_item, $order_item_quantity ) {

			// This function is directly related to reduced_stock_meta_manipulation()

			// This function is hooked to the woocommerce_prevent_adjust_line_item_product_stock filter hook which is used by wc_maybe_adjust_line_item_product_stock() when order items are saved or removed, if the wcrp_rental_products_rent_from key exists on the order item we prevent any purchasable stock changes, e.g. order contains a rental or purchase product (rental only has manage stock disabled so is uneffected), order is set to on hold, order item is removed, without the conditions below the order item would have its _reduced_stock meta got which will exist due to reduced_stock_meta_manipulation() and if exists and > 0 then it would adjust stock accordingly, we do not want this to occur because its a rental and therefore purchasable stock should be unaffected by the removal, this is primarily for on hold orders but also true of cancelled or pending but for those even if the condition below wasn't here those wouldn't get stock changes because upon changing to those statuses the _reduced_stock meta gets removed via reduced_stock_meta_manipulation() - so the conditions in wc_maybe_adjust_line_item_product_stock() after the woocommerce_prevent_adjust_line_item_product_stock filter hook would not do the changes because the _reduced_stock meta is removed

			$order_item_id = $order_item->get_id();

			if ( !empty( $order_item_id ) ) {

				if ( !empty( wc_get_order_item_meta( $order_item_id, 'wcrp_rental_products_rent_from', true ) ) ) {

					$restrict = true;
	
				}

			}

			return $restrict;

		}

	}

}
