<?php

namespace WP_Defender\Model\Notification;

use WP_Defender\Model\Scan;
use WP_Defender\Model\Scan_Item;
use WP_Defender\Traits\IO;
use WP_Defender\Traits\Scan_Email_Template;
use WP_Defender\Model\Notification\Malware_Report;

/**
 * Class Firewall_Notification
 * @package WP_Defender\Model\Notification
 */
class Malware_Notification extends \WP_Defender\Model\Notification {
	use IO, Scan_Email_Template;

	/**
	 * Option name.
	 * @var string
	 */
	protected $table = 'wd_malware_scanning_notification';

	/**
	 * @return array
	 */
	public function get_default_values(): array {
		$template = $this->get_email_template();

		return [
			'subject_issue_found' => $template['found']['subject'],
			'subject_issue_not_found' => $template['not_found']['subject'],
			'subject_error' => $template['error']['subject'],
			'content_issue_found' => $template['found']['body'],
			'content_issue_not_found' => $template['not_found']['body'],
			'content_error' => $template['error']['body'],
		];
	}

	protected function before_load(): void {
		$default = [
			'slug' => 'malware-notification',
			'title' => __( 'Malware Scanning - Notification', 'wpdef' ),
			'status' => self::STATUS_DISABLED,
			'description' => __( 'Get email notifications when Defender has finished manual malware scans.', 'wpdef' ),
			// @since 3.0.0 Fix 'Guest'-line.
			'in_house_recipients' => is_user_logged_in() ? [ $this->get_default_user() ] : [],
			'out_house_recipients' => [],
			'type' => 'notification',
			'dry_run' => false,
			'configs' => [
				'always_send' => false,
				'error_send' => false,
				'template' => $this->get_email_template(),
			],
		];
		$this->import( $default );
	}

	/**
	 * @param Scan   $scan
	 * @param object $object
	 * @param string $subject
	 * @param string $content
	 * @param array  $issues
	 */
	protected function email_engine( Scan $scan, $object, $subject, $content, $issues ): void {
		$service = wd_di()->get( \WP_Defender\Component\Notification::class );
		foreach ( $object->in_house_recipients as $user ) {
			if ( self::USER_SUBSCRIBED !== $user['status'] ) {
				continue;
			}
			$this->send_to_user( $user['email'], $user['name'], $subject, $content, $issues, $service );
		}
		foreach ( $object->out_house_recipients as $user ) {
			if ( self::USER_SUBSCRIBED !== $user['status'] ) {
				continue;
			}
			$this->send_to_user( $user['email'], $user['name'], $subject, $content, $issues, $service );
		}
	}

	/**
	 * @param Scan $scan
	 *
	 * @return void
	 */
	public function send( Scan $scan ) {
		$object = $this;
		if ( $scan->is_automation ) {
			$object = wd_di()->get( Malware_Report::class );
			// If Malware Reporting is unchecked, then return.
			if ( self::STATUS_ACTIVE !== $object->status ) {
				return;
			}
			// @since 2.7.0 Remove 'dry_run'-condition.
		}
		$issues = $scan->get_issues( null, Scan_Item::STATUS_ACTIVE );
		/**
		 * When the Defender is unable to scan files for some reason and the 'error_send' option is checked.
		 * Possible reasons:
		 * - no connection to Scan API,
		 * - no checksums received from wp.org,
		 * - scan process stuck.
		 */
		if (
			isset( $object->configs['error_send'] ) && ! empty(
				$object->configs['error_send']
				&& in_array( $scan->status, [ Scan::STATUS_ERROR, Scan::STATUS_IDLE ], true )
			)
		) {
			$subject = $object->configs['template']['error']['subject'];
			$content = nl2br( $object->configs['template']['error']['body'] );

			return $this->email_engine( $scan, $object, $subject, $content, $issues );
		}

		// Sometimes value may be as empty string for disabled 'always_send'.
		if ( empty( $object->configs['always_send'] ) && 0 === count( $issues ) ) {
			return;
		}
		// For another case:
		$templates = count( $issues ) ? $object->configs['template']['found'] : $object->configs['template']['not_found'];
		$subject = $templates['subject'];
		$content = nl2br( $templates['body'] );

		return $this->email_engine( $scan, $object, $subject, $content, $issues );
	}

	/**
	 * @param $email
	 * @param $name
	 * @param $subject
	 * @param $content
	 * @param $issues
	 * @param $service
	 *
	 * @throws \DI\DependencyException
	 * @throws \DI\NotFoundException
	 */
	private function send_to_user( $email, $name, $subject, $content, $issues, $service ) {
		$controller_scan = wd_di()->get( \WP_Defender\Controller\Scan::class );
		$replacers = [
			'issues_count' => is_array( $issues ) || $issues instanceof \Countable ? (string) count( $issues ) : '0',
			'issues_list' => $controller_scan->render_partial(
				'email/scan-issue-list',
				[
					'issues' => $issues,
					'email' => $email,
				],
				false
			),
			'site_url' => network_site_url(),
			'user_name' => $name,
		];

		/**
		 * Filters the replaced data. It's without replacement.
		 * @deprecated 3.2.0
		 * @param array $replacers Email params.
		 */
		$replacers = apply_filters_deprecated( 'wd_notification_email_params', [ $replacers ], '3.2.0' );

		$filter_name = ( is_array( $issues ) || $issues instanceof \Countable ? count( $issues ) : 0 ) > 0
			? 'wd_notification_email_subject_issue'
			: 'wd_notification_email_subject';
		$subject = apply_filters( $filter_name, $subject );

		/**
		 * It's without replacement.
		 * @deprecated 3.2.0
		 */
		$content = apply_filters_deprecated(
			'wd_notification_email_content_before',
			[ $content ],
			'3.2.0'
		);
		foreach ( $replacers as $key => $replacer ) {

			if ( is_scalar( $replacer ) && ! is_string( $replacer ) ) {
				$replacer = (string) $replacer;
			}

			$content = str_replace( "{{$key}}", $replacer, $content );
			$content = str_replace( '{' . strtoupper( $key ) . '}', $replacer, $content );
			$subject = str_replace( "{{$key}}", $replacer, $subject );
			$subject = str_replace( '{' . strtoupper( $key ) . '}', $replacer, $subject );
		}
		/**
		 * It's without replacement.
		 * @deprecated 3.2.0
		 */
		$content = apply_filters_deprecated(
			'wd_notification_email_content_after',
			[ $content ],
			'3.2.0'
		);
		$unsubscribe_link = $service->create_unsubscribe_url( $this->slug, $email );
		$template = $controller_scan->render_partial(
			'email/index',
			[
				'title' => __( 'Malware Scanning', 'wpdef' ),
				'content_body' => $content,
				'unsubscribe_link' => $unsubscribe_link,
			],
			false
		);

		$headers = defender_noreply_html_header(
			defender_noreply_email( 'wd_scan_noreply_email' )
		);

		$ret = wp_mail( $email, $subject, $template, $headers );
		if ( $ret ) {
			$this->save_log( $email );
		}
	}

	/**
	 * Define settings labels.
	 *
	 * @return array
	 */
	public function labels(): array {
		return [
			'notification' => __( 'Malware Scanning - Notification', 'wpdef' ),
			'always_send_notification' => __( 'Send notifications when no issues are detected', 'wpdef' ),
			'error_send' => __( "Send notifications when Defender couldn't scan the files", 'wpdef' ),
			'email_subject_issue_found' => __( 'Email subject when an issue is found', 'wpdef' ),
			'email_subject_issue_not_found' => __( 'Email subject when no issues are found', 'wpdef' ),
			'email_subject_error' => __( 'Email subject when failed to scan', 'wpdef' ),
			'email_content_issue_found' => __( 'Email body when an issue is found', 'wpdef' ),
			'email_content_issue_not_found' => __( 'Email body when no issues are found', 'wpdef' ),
			'email_content_error' => __( 'When failed to scan', 'wpdef' ),
			'notification_subscribers' => __( 'Recipients', 'wpdef' ),
		];
	}

	/**
	 * Try to send email if:
	 * Malware Scanning - Notification is enabled,
	 * Malware Scanning - Reporting is enabled or Scan model has is_automation = true.
	 *
	 * @return bool
	*/
	public function check_options(): bool {
		return self::STATUS_ACTIVE === $this->status
			|| self::STATUS_ACTIVE === wd_di()->get( Malware_Report::class )->status;
	}

	/**
	 * Additional converting rules.
	 *
	 * @param array $configs
	 *
	 * @return array
	 * @since 3.1.0
	 */
	public function type_casting( array $configs ): array {
		$configs['always_send'] = (bool) $configs['always_send'];
		$configs['error_send'] = (bool) $configs['error_send'];
		// Additional security layer because parent 'configs'-property doesn't have sanitized stuff for nested properties.
		$template = $configs['template'];
		$configs['template']['found']['subject'] = sanitize_text_field( $template['found']['subject'] );
		$configs['template']['found']['body'] = sanitize_textarea_field( $template['found']['body'] );
		$configs['template']['not_found']['subject'] = sanitize_text_field( $template['not_found']['subject'] );
		$configs['template']['not_found']['body'] = sanitize_textarea_field( $template['not_found']['body'] );
		$configs['template']['error']['subject'] = sanitize_text_field( $template['error']['subject'] );
		$configs['template']['error']['body'] = sanitize_textarea_field( $template['error']['body'] );

		return $configs;
	}
}