<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WCRP_Rental_Products_Order_Save' ) ) {

	class WCRP_Rental_Products_Order_Save {

		public function __construct() {

			add_action( 'edit_form_top', array( $this, 'order_save_nonce' ) );
			add_action( 'woocommerce_update_order', array( $this, 'order_save' ) );
			add_action( 'before_delete_post', array(
				$this,
				'order_delete'
			) ); // Not using woocommerce_delete_order due to https://github.com/woocommerce/woocommerce/issues/32543
			add_action( 'woocommerce_order_status_completed', array( $this, 'order_status_completed' ) );
			add_action( 'woocommerce_new_order', array( $this, 'rentals_add_update' ) );
			add_action( 'woocommerce_update_order', array( $this, 'rentals_add_update' ) );
			add_action( 'woocommerce_update_order', array( $this, 'rentals_remove' ) );
			add_action( 'woocommerce_update_order', array( $this, 'reduced_stock_meta_manipulation' ) );

		}

		public function order_save_nonce( $post ) {

			if ( ! empty( $post ) ) {

				if ( 'shop_order' == get_post_type( $post->ID ) ) {

					wp_nonce_field( 'wcrp_rental_products_order_save', 'wcrp_rental_products_order_save_nonce' );

				}

			}

		}

		public function order_save( $order_id ) {

			global $wpdb;

			if ( isset( $_POST['wcrp_rental_products_order_save_nonce'] ) && wp_verify_nonce( sanitize_key( $_POST['wcrp_rental_products_order_save_nonce'] ), 'wcrp_rental_products_order_save' ) ) {

				$order = wc_get_order( $order_id );

				if ( ! empty( $order ) ) {

					if ( isset( $_POST['wcrp_rental_products_line_item_cancel_rental'] ) ) {

						$cancel_rental_order_item_id = sanitize_text_field( $_POST['wcrp_rental_products_line_item_cancel_rental'] );

						$wpdb->query(
							$wpdb->prepare(
								"DELETE FROM `{$wpdb->prefix}wcrp_rental_products_rentals` WHERE `order_item_id` = %d;",
								$cancel_rental_order_item_id
							)
						); // Note that we do not also delete from the archive table as that only contains rentals marked as returned and the cancel rental option ($_POST['wcrp_rental_products_line_item_cancel_rental']) is not available for rental order items marked as returned, it is possible for rentals marked as returned to be cancelled by changing the order status to cancelled but the removal from the archive in this scenario is done via rentals_remove()

						if ( empty( wc_get_order_item_meta( $cancel_rental_order_item_id, 'wcrp_rental_products_cancelled', true ) ) ) {

							$this->delete_order_item_meta_for_cancelled_order_item( $cancel_rental_order_item_id );

							wc_add_order_item_meta( $cancel_rental_order_item_id, 'wcrp_rental_products_cancelled', 'yes', true );

							$this->add_order_note_for_order_item_product( $cancel_rental_order_item_id, 'cancelled' );

						}

					} elseif ( isset( $_POST['wcrp_rental_products_line_item_returned'] ) ) {

						$returned_order_item_id = sanitize_text_field( $_POST['wcrp_rental_products_line_item_returned'] );

						if ( empty( wc_get_order_item_meta( $returned_order_item_id, 'wcrp_rental_products_returned', true ) ) ) {

							wc_add_order_item_meta( $returned_order_item_id, 'wcrp_rental_products_returned', 'yes', true );

							$this->add_order_note_for_order_item_product( $returned_order_item_id, 'returned' );

							// custom query -> add 3 days after item is returned
							$date = date( 'Y-m-d' );
							$product_id   = $order->get_items()[ $returned_order_item_id ]->get_data()['product_id'];
							if ( wc_get_product( $product_id )->is_type( 'variable' ) ) {
								$product_id = $order->get_items()[ $returned_order_item_id ]->get_data()['variation_id'];
							}
							$quantity = $order->get_items()[ $returned_order_item_id ]->get_data()['quantity'];

							//3 days intermission
							for ($i = 0; $i < 3; $i++) {
								$wpdb->query(
									$wpdb->prepare(
										"INSERT INTO `{$wpdb->prefix}wcrp_rental_products_rentals` (reserved_date, order_id, order_item_id, product_id, quantity) VALUES (%s, 0, 0, %d, %d);",
										$date,
										$product_id,
										$quantity,
									)
								);
								//rent the current day as well -> may need to change later
								$date = date('Y-m-d', strtotime($date . ' +1 day'));
							}
						}

					}

					$this->rentals_add_update( $order_id );

				}

			}

		}

		public function order_delete( $order_id ) {

			global $wpdb;

			if ( 'shop_order' == get_post_type( $order_id ) ) {

				$wpdb->query(
					$wpdb->prepare(
						"DELETE FROM `{$wpdb->prefix}wcrp_rental_products_rentals` WHERE `order_id` = %d;",
						$order_id
					)
				);

				$wpdb->query(
					$wpdb->prepare(
						"DELETE FROM `{$wpdb->prefix}wcrp_rental_products_rentals_archive` WHERE `order_id` = %d;",
						$order_id
					)
				);

			}

		}

		public function order_status_completed( $order_id ) {

			$order = wc_get_order( $order_id );

			if ( ! empty( $order ) ) {

				$order_items = $order->get_items();

				if ( ! empty( $order_items ) ) {

					foreach ( $order_items as $order_item ) {

						$order_item_id = $order_item->get_id();

						if ( ! empty( wc_get_order_item_meta( $order_item_id, 'wcrp_rental_products_rent_from', true ) ) ) { // If it is a rental and not cancelled (as if cancelled would not have it)

							if ( 'yes' == get_option( 'wcrp_rental_products_return_rentals_in_completed_orders' ) ) { // If completed order status should return all rentals in order

								if ( empty( wc_get_order_item_meta( $order_item_id, 'wcrp_rental_products_returned', true ) ) ) {

									wc_add_order_item_meta( $order_item_id, 'wcrp_rental_products_returned', 'yes', true );

									$this->add_order_note_for_order_item_product( $order_item_id, 'returned' );

								}

							}

						}

					}

				}
			}

		}

		public static function rentals_add_update( $order_id ) {

			if ( ! empty( $order_id ) ) {

				global $wpdb;

				$order = wc_get_order( $order_id );

				if ( ! empty( $order ) ) {

					$cancel_rentals_in_failed_orders = get_option( 'wcrp_rental_products_cancel_rentals_in_failed_orders' );

					if ( 'yes' == $cancel_rentals_in_failed_orders ) {

						$add_update_order_statuses = array( 'pending', 'processing', 'on-hold' );

					} else {

						$add_update_order_statuses = array( 'pending', 'processing', 'on-hold', 'failed' );

					}

					require_once ABSPATH . 'wp-admin/includes/plugin.php';

					if ( is_plugin_active( 'deposits-partial-payments-for-woocommerce/start.php' ) || is_plugin_active( 'deposits-partial-payments-for-woocommerce-pro/start.php' ) ) {

						$add_update_order_statuses[] = 'partially-paid'; // If Deposits & Partial Payments for WooCommerce is active then partially paid order status is included to ensure rentals are reserved (see managing rental orders meta box for details), this has been added for completeness, however without this it would reserve anyway because created orders always initially get a pending status

					}

					if ( in_array( $order->get_status(), $add_update_order_statuses ) ) {

						$order_items = $order->get_items();

						if ( ! empty( $order_items ) ) {

							foreach ( $order_items as $order_item ) {

								$order_item_id         = $order_item->get_id();
								$order_item_meta_data  = $order_item->get_meta_data();
								$rent_from             = '';
								$rent_to               = '';
								$return_days_threshold = '';
								$returned              = '';
								$cancelled             = '';

								if ( ! empty( $order_item_meta_data ) ) {

									foreach ( $order_item_meta_data as $order_item_meta_data_object ) {

										if ( isset( $order_item_meta_data_object->key ) ) {

											if ( 'wcrp_rental_products_rent_from' == $order_item_meta_data_object->key ) {

												$rent_from = $order_item_meta_data_object->value;

											} elseif ( 'wcrp_rental_products_rent_to' == $order_item_meta_data_object->key ) {

												$rent_to = $order_item_meta_data_object->value;

											} elseif ( 'wcrp_rental_products_return_days_threshold' == $order_item_meta_data_object->key ) {

												$return_days_threshold = $order_item_meta_data_object->value;

											} elseif ( 'wcrp_rental_products_returned' == $order_item_meta_data_object->key ) {

												$returned = $order_item_meta_data_object->value;

											} elseif ( 'wcrp_rental_products_cancelled' == $order_item_meta_data_object->key ) {

												$cancelled = $order_item_meta_data_object->value;

											}

										}

									}

								}

								if ( 'yes' == $returned || 'yes' == $cancelled ) {

									// Rental not reserved for order item if has been returned or cancelled, if returned the database table data remains until archived, if cancelled it has already been removed
									// No check for if rent from/to exists because cancelled do not have it

									continue; // Continue to next order item iteration

								} else {

									// Reserve rental for order item, deletes existing data for this order item then add (adds or updates), the deletion ensures duplicate rows are not inserted, note that the code below is not done for the archive table, as the archive table only includes rentals marked as returned

									if ( '' !== $rent_from && '' !== $rent_to ) { // Rental order items only, without this non-rentals would get reserved

										if ( ! empty( $order_item->get_variation_id() ) ) {

											$product_id = $order_item->get_variation_id();

										} else {

											$product_id = $order_item->get_product_id();

										}

										$quantity = (int) $order_item->get_quantity();

										$wpdb->query(
											$wpdb->prepare(
												"DELETE FROM `{$wpdb->prefix}wcrp_rental_products_rentals` WHERE `order_id` = %d AND `order_item_id` = %d;",
												$order_id,
												$order_item_id
											)
										);

										$begin             = new DateTime( $rent_from );
										$end_modifier_days = (int) $return_days_threshold + 1; // Adds extra for return days plus 1 as 1 day short without
										$end               = new DateTime( gmdate( 'Y-m-d', strtotime( $rent_to . ' +' . $end_modifier_days . ' days' ) ) ); // Specifically not translatable as part of PHP date calculation
										$interval          = DateInterval::createFromDateString( '1 day' );
										$period            = new DatePeriod( $begin, $interval, $end );

										foreach ( $period as $reserved_date ) {

											$wpdb->query(
												$wpdb->prepare(
													"INSERT INTO `{$wpdb->prefix}wcrp_rental_products_rentals` (`rental_id`, `reserved_date`, `order_id`, `order_item_id`, `product_id`, `quantity`) VALUES ( '', %s, %d, %d, %d, %d );",
													$reserved_date->format( 'Y-m-d' ),
													$order_id,
													$order_item_id,
													$product_id,
													$quantity
												)
											);

										}

									}

								}

							}

						}

					}

				}

			}

		}

		public function rentals_remove( $order_id ) {

			$order = wc_get_order( $order_id );

			if ( ! empty( $order ) ) {

				$cancel_rentals_in_failed_orders = get_option( 'wcrp_rental_products_cancel_rentals_in_failed_orders' );

				if ( 'yes' == $cancel_rentals_in_failed_orders ) {

					$removal_order_statuses = array( 'cancelled', 'refunded', 'failed' );

				} else {

					$removal_order_statuses = array( 'cancelled', 'refunded' );

				}

				if ( in_array( $order->get_status(), $removal_order_statuses ) ) {

					global $wpdb;

					$wpdb->query(
						$wpdb->prepare(
							"DELETE FROM `{$wpdb->prefix}wcrp_rental_products_rentals` WHERE `order_id` = %d;",
							$order_id
						)
					);

					$wpdb->query(
						$wpdb->prepare(
							"DELETE FROM `{$wpdb->prefix}wcrp_rental_products_rentals_archive` WHERE `order_id` = %d;",
							$order_id
						)
					);

					$order_items = $order->get_items();

					if ( ! empty( $order_items ) ) {

						// Rental order item data must be removed, this ensures that if the order status is changed to a status which would attempt to rebook the rental it does not happen (as the stock may have since been used by another order and there would be an overlap)

						foreach ( $order_items as $order_item_id => $order_item ) {

							// If it is a rental line item then add cancelled meta (without this non rentals would get the cancelled meta)

							if ( ! empty( wc_get_order_item_meta( $order_item_id, 'wcrp_rental_products_rent_from', true ) ) ) { // If line item is a rental

								if ( empty( wc_get_order_item_meta( $order_item_id, 'wcrp_rental_products_cancelled', true ) ) ) {

									$this->delete_order_item_meta_for_cancelled_order_item( $order_item_id );

									wc_add_order_item_meta( $order_item_id, 'wcrp_rental_products_cancelled', 'yes', true );

									$this->add_order_note_for_order_item_product( $order_item_id, 'cancelled' );

								}

							}

						}

					}

				}

			}

		}

		public function reduced_stock_meta_manipulation( $order_id ) {

			// This function is directly related to order_line_item_restrict_purchasable_stock_adjustment_if_rental()

			if ( ! empty( $order_id ) ) {

				$order = wc_get_order( $order_id );

				if ( ! empty( $order ) ) {

					$order_items  = $order->get_items();
					$order_status = $order->get_status();

					if ( ! empty( $order_items ) ) {

						foreach ( $order_items as $order_item ) {

							$order_item_id        = $order_item->get_id();
							$order_item_meta_data = $order_item->get_meta_data();

							if ( ! empty( $order_item_meta_data ) ) {

								foreach ( $order_item_meta_data as $order_item_meta_data_object ) {

									if ( isset( $order_item_meta_data_object->key ) ) {

										// If order item is a rental, its fine this occurs for all rental products not just rental or purchase, _reduced_stock would only be used for rental or purchase products which have stock management enabled as rental only have stock management disabled so even if this meta is there it isn't used, also we don't check for whether stock management in general is enabled, this is just because the meta below should still be set so if stock management enabled in future it doesn't amend any stock due to it being there/not there during the period it was not enabled

										if ( 'wcrp_rental_products_rent_from' == $order_item_meta_data_object->key ) {

											if ( in_array( $order_status, array( 'cancelled', 'pending' ) ) ) {

												// If the order status is one of the above then _reduced_stock is deleted, this is because if it remains then the value will be added back to the purchasable stock, the wc_maybe_increase_stock_levels() function is hooked off of the woocommerce_order_status_cancelled and woocommerce_order_status_pending action hooks, that function gets the _reduced_stock and adds it, so if changed to this status we remove it so the increase does not occur e.g. rental or purchase based rental is in the order, order is processing so _reduced_stock is the qty, user then changes order status to cancelled/pending, wc_maybe_increase_stock_levels() reads the _reduced_stock to see if it needs to increase, the below removes it before then so it doesn't get increased

												wc_delete_order_item_meta( $order_item_id, '_reduced_stock' );

											} else {

												if ( empty( wc_get_order_item_meta( $order_item_id, '_reduced_stock', true ) ) ) {

													// This sets _reduced_stock to the quantity, this is done to tell WooCommerce that this rental order line item has had purchasable stock reduced, this meta is used by WooCommerce in functions like wc_reduce_stock_levels() and if set then it doesn't reduce the stock levels, this needs to be done as if the product is a rental or purchase product and it is the rental then if, for example, an order status is paid or set to processing and other statuses (e.g. see the actions hooks used for wc_maybe_reduce_stock_levels which in turn calls wc_reduce_stock_levels() which checks _reduced_stock) then it doesn't reduce the purchasable stock of the rental or purchase product

													wc_add_order_item_meta( $order_item_id, '_reduced_stock', $order_item->get_quantity(), true );

												}

											}

											break;

										}

									}

								}

							}

						}

					}

				}

			}

		}

		public function add_order_note_for_order_item_product( $order_item_id, $note_type ) {

			if ( ! empty( $order_item_id ) && ! empty( $note_type ) ) {

				$order_item = new WC_Order_Item_Product( $order_item_id );

				if ( ! empty( $order_item ) ) {

					$order = $order_item->get_order();

					if ( ! empty( $order ) ) {

						$order_item_product      = $order_item->get_product();
						$order_item_product_name = $order_item->get_name();

						if ( ! empty( $order_item_product ) ) {

							$order_item_product_sku = $order_item_product->get_sku();

							if ( 'cancelled' == $note_type ) {

								$order->add_order_note(
									esc_html__( 'Rental cancelled:', 'wcrp-rental-products' ) . ' ' . $order_item_product_name . ( ! empty( $order_item_product_sku ) ? ' ' . esc_html__( '(', 'wcrp-rental-products' ) . $order_item_product_sku . esc_html__( ')', 'wcrp-rental-products' ) : '' ) . esc_html__( '.', 'wcrp-rental-products' ),
									false,
									true
								);

							} elseif ( 'returned' == $note_type ) {

								$order->add_order_note(
									esc_html__( 'Rental marked as returned:', 'wcrp-rental-products' ) . ' ' . $order_item_product_name . ( ! empty( $order_item_product_sku ) ? ' ' . esc_html__( '(', 'wcrp-rental-products' ) . $order_item_product_sku . esc_html__( ')', 'wcrp-rental-products' ) : '' ) . esc_html__( '.', 'wcrp-rental-products' ),
									false,
									true
								);

							}

						}

					}

				}

			}

		}

		public function delete_order_item_meta_for_cancelled_order_item( $order_item_id ) {

			if ( ! empty( $order_item_id ) ) {

				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_rent_from' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_rent_to' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_return_days_threshold' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_returned' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_in_person_pick_up_return' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_in_person_pick_up_date' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_in_person_pick_up_time' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_in_person_pick_up_fee' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_in_person_return_date' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_in_person_return_date_type' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_in_person_return_time' );
				wc_delete_order_item_meta( $order_item_id, 'wcrp_rental_products_in_person_return_fee' );

			}

		}

	}

}
